/**
* \file: mounter.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: startup
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/

#include "mounter.h"

#include <sys/mount.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <pwd.h>
#include <grp.h>

#include <assert.h>
#include <errno.h>

#include <errno.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#include <alloca.h>
#include <string.h>


//defines the access mode of mount point folders created by the mounter module
#define MP_MODE 					0777

//Mount options taken from a configuration file must be modified (user -> user id, groups -> group id).
//It might happen that the extracted options string is larger than the one read out from file. So a new byte array
//is allocated for the modified options string. The size of the new array is calculated as follows:
//new_size=strlen(raw_options_string)+ADD_OPTIONS_BUFFER_SIZE
//The buffer must be large enough to take the complete modified option string
//Currently, only the names in 'gid=<group name>' and 'uid=<user name>' are replaced by the respective ids. With
//sizeof(gid_t) and sizeof(uid_t) = 32bit, a buffer extension of 256 bytes should be enough
#define DATA_BUFFER_SIZE_EXTENSION 	256

//-------------------------------------- lookup table with mount options to be replaced by flags ----------------------
typedef struct mount_flag_map_t
{
	const char *option;
	unsigned long flag;
} mount_flag_map_t;

static const mount_flag_map_t invalid_parameter_list[]=
{
		{"exec",0},
		{"noexec",MS_NOEXEC},
		{"suid",0},
		{"nosuid",MS_NOSUID},
		{"relatime",MS_RELATIME},
		{"norelatime",0},
		{"nodev",MS_NODEV},
		{"ro",MS_RDONLY},
		{"rw",0},
		{NULL,0}
};
//---------------------------------------------------------------------------------------------------------------------

//-------------------------------------- private member declaration ---------------------------------------------------
static bool mounter_does_mp_exist(const char *mp);
static error_code_t mounter_create_mp(const char *mp);
static error_code_t mounter_extract_sring_element(int line_no, char *token, char **data_struct_element,
		const char *descr);
static error_code_t mounter_extract_flags_and_mount(int line_no, mount_point_t *mp, char *raw_options);
static error_code_t mounter_check_mount_option(int line_no, char *option, char **data_buf_write_ptr,
		const char *data_buf_end, unsigned long *flags_ptr);
static bool mounter_is_key_value_option(const char *option, const char *searched_key, size_t key_len);
static error_code_t mounter_process_uid(int line_no, char *option, char **data_buf_write_ptr, const char *data_buf_end);
static error_code_t mounter_process_gid(int line_no, char *option, char **data_buf_write_ptr, const char *data_buf_end);
static void mounter_check_and_add_string_to_data(char **data_buf_write_ptr, const char *data_buf_end,
		const char *option2add);
static error_code_t mounter_get_gid_from_string(int line_no, const char *gid_string,gid_t *id_ptr);
static error_code_t mounter_get_uid_from_string(int line_no, const char *uid_string,uid_t *id_ptr);
static error_code_t mounter_process_normal_option(const char *option, char **data_buf_write_ptr,
		const char *data_buf_end, unsigned long *flags_ptr);
static char *mounter_trim_string(char *string);
//---------------------------------------------------------------------------------------------------------------------

//-------------------------------------- public member definition ----------------------------------------------------
error_code_t mounter_do_mount(const mount_point_t *mp)
{
	error_code_t result=RESULT_OK;
	bool does_mp_exist;

	does_mp_exist=mounter_does_mp_exist(mp->mp);

	if (!does_mp_exist)
	{
		if (mp->create_mp)
		{
			result=mounter_create_mp(mp->mp);
		}
		else
		{
			fprintf(stderr, "Mount point \"%s\" does not exist and shall not be created according to configuration.\n",
					mp->mp);
			result=RESULT_NO_MOUNT_POINT;
		}
	}

	if (result==RESULT_OK)
	{
		if (mount(mp->src,mp->mp,mp->fs,mp->flags,mp->data)!=0)
		{
			fprintf(stderr, "Unable to mount \"%s\": %s\n",mp->mp,strerror(errno));
			fprintf(stderr, "Mount point: %s\n",mp->mp);
			fprintf(stderr, "Mount source: %s\n",mp->src);
			fprintf(stderr, "File system: %s\n",mp->fs);
			fprintf(stderr, "Options: %s\n",mp->data);
			fprintf(stderr, "Flags: 0x%lX\n",mp->flags);
			result=RESULT_MOUNT_ERR;
		}
	}

	return result;
}

error_code_t mounter_do_parse_and_mount(int line_no, char* conf_file_line)
{
	error_code_t result;
	mount_point_t mp;
	char* token;
	//options read out from conf file
	char* raw_options=NULL;

	token=strtok(conf_file_line, ":\n");

	//mount point
	result=mounter_extract_sring_element(line_no,token, &(mp.mp), "mount point");

	//source
	if (result==RESULT_OK)
	{
		token=strtok(NULL, ":\n");
		result=mounter_extract_sring_element(line_no,token, &(mp.src), "source");
	}

	//file system
	if (result==RESULT_OK)
	{
		token=strtok(NULL, ":\n");
		result=mounter_extract_sring_element(line_no,token, &(mp.fs), "file system");
	}

	//options
	if (result==RESULT_OK)
	{
		token=strtok(NULL, ":\n");
		//options is optional
		if (token != NULL)
			result=mounter_extract_sring_element(line_no,token, &raw_options, "options");
	}

	//do the actual mounting
	if (result==RESULT_OK)
		result=mounter_extract_flags_and_mount(line_no, &mp, raw_options);

	return result;
}
//---------------------------------------------------------------------------------------------------------------------

//-------------------------------------- private member definition ----------------------------------------------------
static error_code_t mounter_create_mp(const char *mp)
{
	char *mp_clone;
	char *search_ptr;
	size_t len;

	assert(mp != NULL);

	//check if mp is an absolute path to prevent that we are creating a relative somewhere
	if (mp[0]!='/')
	{
		fprintf(stderr, "Mount point \"%s\" is not an valid absolute path.\n",mp);
		return RESULT_INVALID_MP_PATH;
	}

	//does anyone pass us "/" as path?
	len=strlen(mp);
	if (len==1)
		return RESULT_OK;

	//create a clone of the path, we want to modify it
	mp_clone=(char *)alloca(len+1);
	strcpy(mp_clone, mp);

	//if our path ends on '/', remove it
	if (mp_clone[len-1]=='/')
		mp_clone[len-1]='\0';

	//the len check above ensures that we have at least 2 characters in the path
	search_ptr=mp_clone+1;

	//create the path component by component
	do
	{
		//find the next '/' in path
		search_ptr=strchr(search_ptr, '/');

		//cat the path at the position of the next '/' by just inserting '\0'
		if (search_ptr!=NULL)
			*search_ptr='\0';

		//create the part of the path if it does not exist
		if (!mounter_does_mp_exist(mp_clone))
		{
			if (mkdir(mp_clone, MP_MODE)!=0)
			{
				fprintf(stderr, "Unable to create mount point \"%s\": %s\n",mp,strerror(errno));
				return RESULT_MP_CREATION_FAILED;
			}
		}

		//put back the '/' and move to the next character to search for the next '/' in the next round
		if (search_ptr!=NULL)
		{
			*search_ptr='/';
			search_ptr++;

		}
	}
	while(search_ptr != NULL);

	return RESULT_OK;
}

static bool mounter_does_mp_exist(const char *mp)
{
	struct stat stat_result;
	int result;
	result=stat(mp, &stat_result);
	if (result!=0)
		return false;
	return S_ISDIR(stat_result.st_mode);
}

static error_code_t mounter_extract_sring_element(int line_no, char *token, char **data_struct_element, const char *descr)
{
	token=mounter_trim_string(token);
	//be careful with this function. No copying is done. The lifetime of data_struct_element depends on the lifetime
	//of the token
	if (token == NULL)
	{
		fprintf(stderr, "Line %d invalid in configuration file: %s expected\n",line_no, descr);
		fprintf(stderr, "Expected format per line: <mount point>:<source>:<file system>:<options>\n\n");
		return RESULT_USAGE;
	}

	*data_struct_element=token;
	return RESULT_OK;
}

static error_code_t mounter_extract_flags_and_mount(int line_no, mount_point_t *mp, char *raw_options)
{
	//buffer for the new data string
	char *data_buf;
	//len of the buffer
	size_t data_buffer_size;

	//cursor and end marker
	char *data_buf_write_ptr;
	char *data_buf_end;

	char *token=NULL;

	error_code_t result=RESULT_OK;

	//reset flags
	mp->flags=0;

	//allocate buffer for the new data string
	if (raw_options!=NULL)
		data_buffer_size=DATA_BUFFER_SIZE_EXTENSION+strlen(raw_options);
	else
		data_buffer_size=1;

	data_buf=(char *)alloca(data_buffer_size);

	//lets make it an empty string
	data_buf[0]='\0';

	//and position the cursor
	data_buf_write_ptr=data_buf;
	data_buf_end=data_buf+data_buffer_size-1;

	//start parsing the given options string
	if (raw_options!=NULL)
		token=strtok(raw_options, ",");
	while(token != NULL && result==RESULT_OK)
	{
		token=mounter_trim_string(token);
		result=mounter_check_mount_option(line_no, token, &data_buf_write_ptr, data_buf_end, &(mp->flags));
		token=strtok(NULL,",");
	}

	if (result==RESULT_OK)
	{
		//in case we have added any option, we are starting with ','. Lets skip it.
		if (data_buf[0]==',')
			mp->data=data_buf+1;
		else
			mp->data=data_buf;
		result=mounter_do_mount(mp);
	}

	return result;
}

static error_code_t mounter_check_mount_option(int line_no, char *option, char **data_buf_write_ptr, const char *data_buf_end,
		unsigned long *flags_ptr)
{
	if (mounter_is_key_value_option(option, "gid", sizeof("gid")-1))
		return mounter_process_gid(line_no, option, data_buf_write_ptr, data_buf_end);
	else if (mounter_is_key_value_option(option, "uid", sizeof("uid")-1))
		return mounter_process_uid(line_no, option, data_buf_write_ptr, data_buf_end);
	else
		return mounter_process_normal_option(option, data_buf_write_ptr, data_buf_end, flags_ptr);
}

static bool mounter_is_key_value_option(const char *option, const char *searched_key, size_t key_len)
{
	// the tokenizer ensures that we are trimmed left side
	if (strstr(option,searched_key)!=option)
		return false;

	//after key, we are accepting only spaces & tabs until '='
	option+=key_len;

	while(*option==' ' || *option=='\t')
		option++;

	return *option=='=';
}

static error_code_t mounter_process_uid(int line_no, char *option, char **data_buf_write_ptr, const char *data_buf_end)
{
	char *uid_ptr;
	//should be enough to take 'uid=<uid_t>', with sizeof(uid_t) = 32bit
	char final_uid_option[128];
	uid_t uid;
	error_code_t result;

	uid_ptr=strchr(option, '=');
	if (uid_ptr==NULL)
	{
		fprintf(stderr, "Invalid options in line %d: %s\n", line_no, option);
		return RESULT_INVALID_MOUNT_OPTION;
	}
	uid_ptr++;
	uid_ptr=mounter_trim_string(uid_ptr);

	result=mounter_get_uid_from_string(line_no, uid_ptr, &uid);
	if (result==RESULT_OK)
	{
		if (snprintf(final_uid_option,sizeof(final_uid_option), "uid=%u",uid)>(ssize_t)sizeof(final_uid_option))
		{
			fprintf(stderr, "Buffer needed to take final uid option not large enough. Increase buffer size internally.\n");
			assert(false);
		}
		else
			mounter_check_and_add_string_to_data(data_buf_write_ptr, data_buf_end, final_uid_option);
	}

	return result;
}

static error_code_t mounter_process_gid(int line_no, char *option, char **data_buf_write_ptr, const char *data_buf_end)
{
	char *gid_ptr;
	//should be enough to take 'gid=<gid_t>', with sizeof(gid_t) = 32bit
	char final_gid_option[128];
	gid_t gid;
	error_code_t result;

	gid_ptr=strchr(option, '=');
	if (gid_ptr==NULL)
	{
		fprintf(stderr, "Invalid options in line %d: %s\n",line_no, option);
		return RESULT_INVALID_MOUNT_OPTION;
	}
	gid_ptr++;

	gid_ptr=mounter_trim_string(gid_ptr);

	result=mounter_get_gid_from_string(line_no, gid_ptr, &gid);
	if (result==RESULT_OK)
	{
		if (snprintf(final_gid_option,sizeof(final_gid_option), "gid=%u",gid)>(ssize_t)sizeof(final_gid_option))
		{
			fprintf(stderr, "Buffer needed to take final gid option not large enough. Increase buffer size internally.\n");
			assert(false);
		}
		else
			mounter_check_and_add_string_to_data(data_buf_write_ptr, data_buf_end, final_gid_option);
	}

	return result;
}

static void mounter_check_and_add_string_to_data(char **data_buf_write_ptr, const char *data_buf_end, const char *option2add)
{
	size_t option2add_len;
	ssize_t left_in_buffer;

	//','+option + null terminating byte
	option2add_len=strlen(option2add)+2;
	left_in_buffer = (ssize_t)(data_buf_end-(*data_buf_write_ptr));

	if (left_in_buffer < (ssize_t)option2add_len)
	{
		fprintf(stderr, "Unable to add option \"%s\" to the final mount data string. Increase buffer size internally.\n",
				option2add);
		assert(false);
	}
	//add ','
	**data_buf_write_ptr=',';
	(*data_buf_write_ptr)++;

	memcpy(*data_buf_write_ptr, option2add, option2add_len-1);
	//point to the null terminating byte
	*data_buf_write_ptr+=(option2add_len-2);
}

static error_code_t mounter_get_gid_from_string(int line_no, const char *gid_string,gid_t *id_ptr)
{
	struct group *group_lookup_result;
	char *parse_result;
	//try first to parse for a number which is assumed to be the id
	*id_ptr=(gid_t)strtol(gid_string, &parse_result, 10);

	if (parse_result!=gid_string)
		return RESULT_OK;

	errno=0;
	//given value is not a number, take the value as name and determine the group id
	group_lookup_result=getgrnam (gid_string);

	if (group_lookup_result!=NULL)
	{
		*id_ptr=group_lookup_result->gr_gid;
		return RESULT_OK;
	}
	else
	{
		if (errno==0)
		{
			fprintf(stderr, "Error in line %d: Group \"%s\" not found in the system.\n", line_no, gid_string);
			return RESULT_INVALID_MOUNT_OPTION;
		}
		else
		{
			fprintf(stderr, "Error in line %d: Unable to lookup group \"%s\": %s\n", line_no, gid_string,strerror(errno));
			return RESULT_INVALID_MOUNT_OPTION;
		}
	}
}

static error_code_t mounter_get_uid_from_string(int line_no, const char *uid_string,uid_t *id_ptr)
{
	struct passwd *user_lookup_result;
	char *parse_result;
	//try first to parse for a number which is assumed to be the id
	*id_ptr=(uid_t)strtol(uid_string, &parse_result, 10);

	if (parse_result!=uid_string)
		return RESULT_OK;

	errno=0;
	//given value is not a number, take the value as name and determine the group id
	user_lookup_result=getpwnam (uid_string);

	if (user_lookup_result!=NULL)
	{
		*id_ptr=user_lookup_result->pw_uid;
		return RESULT_OK;
	}
	else
	{
		if (errno==0)
		{
			fprintf(stderr, "Error in line %d: User \"%s\" not found in the system.\n", line_no, uid_string);
			return RESULT_INVALID_MOUNT_OPTION;
		}
		else
		{
			fprintf(stderr, "Error in line %d: Unable to lookup user \"%s\": %s\n", line_no, uid_string,strerror(errno));
			return RESULT_INVALID_MOUNT_OPTION;
		}
	}
}

static error_code_t mounter_process_normal_option(const char *option, char **data_buf_write_ptr,
		const char *data_buf_end, unsigned long *flags_ptr)
{
	const mount_flag_map_t *list_entry;
	list_entry=invalid_parameter_list;
	while (list_entry->option!=NULL)
	{
		if (strcmp(option,list_entry->option)==0)
		{
			*flags_ptr |= list_entry->flag;
			return RESULT_OK;
		}
		list_entry++;
	}
	//not found
	mounter_check_and_add_string_to_data(data_buf_write_ptr,data_buf_end,option);

	return RESULT_OK;
}

static char *mounter_trim_string(char *string)
{
	size_t len;
	char* p_begin;
	char* p_end;

	assert(string != NULL);

	len=strlen(string);

	p_begin=string;
	//end points to the character before the terminating '\0'
	p_end=string+len-1;

	//remove leading spaces
	//each string is terminated by '\0' so this loop will end
	while((*p_begin)==' ')
		p_begin++;

	//remove trailing spaces or new lines or tabs
	while(p_end >= p_begin && ((*p_end)=='\n' || (*p_end)==' ' || (*p_end)=='\t'))
	{
		(*p_end)='\0';
		p_end--;
	}

	return p_begin;
}
//---------------------------------------------------------------------------------------------------------------------
